﻿' ESCV --- © 2007-2022 Mario Corsolini --- https://www.oipaz.net/ESCV.html
' Distributed under the Creative Commons Attribution-NonCommercial-ShareAlike 4.0
' International licence (CC BY-NC-SA 4.0) https://creativecommons.org/licenses/by-nc-sa/4.0/
'
' This module must be part of a "WPF App (.NET Framework)" project.
' It has been tested with "Visual Studio 2022", ".NET Framework 4.8"
' and "Emgu CV 4.1.1".
' 
' The module exposes three main public functions:
' [1] CreateMarksDistribution
'     Input parameters:
'         CSpan (width of each class of marks, but the last one);
'         ImW (width in pixels of the image to be created);
'         MDColumns (data of the histogram).
'     It  returns  a  System.Drawing.Bitmap containing the requested
'     histogram.
' [2] LoadAnswersFromCamera
'     Input parameters:
'         TestID  (one  character  for  term  number,  one  dot, two
'             characters for the number of the test in the term);
'         Title (string to be displayed in the output window).
'     If something goes wrong it returns a single error message.
'     If  it  is  successful it returns two strings: a report of the
'     results and the ID of the questionnaire found.
' [3] LoadAnswersFromLocalImages
'     Input parameters:
'         TestID  (one  character  for  term  number,  one  dot, two
'             characters for the number of the test in the term).
'     If something goes wrong it returns a single error message.
'     If  it  is  successful it returns two strings: a report of the
'     results and the list of unrecognised files (if there are any).
' All   the   other   functions  and  subs  are  used  for  internal
' computations  (even those declared as Public as they are needed by
' other modules of ESCV).
' The  first  function of the module (D) must be enabled or replaced
' by  a  function that translates the content of the input parameter
' into other languages.
' 
' This  module also requires a number of properties to be set before
' its  functions are called.  They are in Options (a public instance
' of the class ProgramOptions) and in Test (a public instance of the
' class TestData, in which the results of the LoadAnswers* functions
' will   be  stored).   You  may  set  the  required  values  either
' programmatically or by loading, respectively, the files "ESCV.ini"
' and  "TestData.xml"  created  by  ESCV.  This is an example of the
' code  required  to  load Test from the file whose complete path is
' stored  in  the  string  FileName (the same code may be adapted to
' load Options, as "ESCV.ini" contains XML data as well):
' Using file As New IO.StreamReader(FileName)
'     Test = CType(New Xml.Serialization.XmlSerializer(GetType(TestData)).Deserialize(file), TestData)
' End Using
'

Imports Emgu.CV
Imports Emgu.CV.Structure
Imports System.Collections.ObjectModel
Imports System.ComponentModel

Public Module modEmguCV

    Public Const AnswerChars As String = "ABCDEFGHIJKLMNOPQRSTUVWXYZ" 'do not alter
    Public Const MainTitle As String = "ESCV by Mario Corsolini"
    Public Const MaxQuestionnaireID As Integer = 999
    Public Const NetTimeout As Integer = 15000
    Public Const T_CancelRadius As Double = 1.25
    Public Const T_DeltaCancel As Double = 1
    Public Const T_DeltaIDBits As Double = 3
    Public Const T_IDBaseShift As Double = 2
    Public Const T_Margins As Double = 12.7
    Public Const T_MaxQuestionsPerColumn As Integer = 5 * (CInt(T_TextHeight / T_DeltaQuestions) \ 5)
    Public Const T_QuestionnaireIDBits As Integer = 10 'it must be a triangular number
    Public Const T_Sleep As Integer = 149
    Public Const T_SquareSide As Double = 5
    Public Const T_TextHeight As Double = T_TotalHeight - 2 * T_Margins
    Public Const T_TextWidth As Double = T_TotalWidth - 2 * T_Margins

    Private Const T_DeltaAnswers As Double = 7.5
    Private Const T_DeltaQuestions As Double = 12.028
    Private Const T_HOffset As Double = -1.3
    Private Const T_NumberWidth As Double = 10
    Private Const T_TotalHeight As Double = 297 'longer side
    Private Const T_TotalWidth As Double = 210 'shorter side
    Private Const T_VOffset As Double = -1.95

    'Public Options As New ProgramOptions
    Public PiQ() As Double = {0, Math.PI / 4, Math.PI / 2, 3 * Math.PI / 4}
    Public T_CameraStatus As Integer = CameraStatus.SwitchedOff
    Public T_ImagesSource As Integer = -3 '-2 (local) or -1 (Android) or >=0 (camera ID number)
    Public T_PowerHistogramChanged As Boolean = False
    'Public Test As New TestData

    Private T_CaptureZ As VideoCapture = Nothing
    Private T_Grey4bits As New Matrix(Of Byte)(256, 1) 'converter to 16-levels grey scale
    Private T_PowerHistogram As New Matrix(Of Byte)(256, 1) 'contrast filter

    Public Enum CameraStatus
        SwitchedOff
        Running
        ShuttingDown
    End Enum

    Public Structure LaTeXdata
        Public Columns As Integer
        Public ColumnWidth As Double
        Public DeltaColumns As Double
        Public QuestionsPerColumn As Integer
        Public TotalLines As Integer
    End Structure

    'this function should translate ENstring into other languages: enable it or replace it with an appropriate one
    'Private Function D(ByVal ENstring As String, Optional ByVal Language As Integer = -1) As String
    '    Return ENstring
    'End Function

    Private Function ActivateCamera() As String

        If T_ImagesSource < 0 Then Return ""
        If Not IsNothing(T_CaptureZ) Then T_CaptureZ.Dispose()
        Try
            T_CaptureZ = New VideoCapture(T_ImagesSource)
            T_CaptureZ.SetCaptureProperty(CvEnum.CapProp.FrameWidth, 1920)
            T_CaptureZ.SetCaptureProperty(CvEnum.CapProp.FrameHeight, 1080)
            Return "OK"
        Catch ex As Exception
            Return D("It is not possible to start:" & vbNewLine & "  ""$$$"".").Replace("$$$", Options.ImagesSource) & vbNewLine & ex.Message
        End Try

    End Function

    Private Sub Autogamma(ByRef Grey As Image(Of Gray, Byte))

        Dim MeanColour As Gray = Grey.GetAverage()

        Grey._GammaCorrect(Math.Log(0.5) / Math.Log(MeanColour.Intensity / 255))

    End Sub

    Public Function CreateMarksDistribution(ByVal CSpan As Double, ByVal ImW As Integer, ByVal MDColumns() As Integer) As System.Drawing.Bitmap

        Dim CN As Integer = MDColumns.Count
        Dim ColumnLabel As String
        Dim ColumnValue As String
        Dim i As Integer
        Dim ImColor As MCvScalar
        Dim ImColorBlack As New MCvScalar(0, 0, 0)
        Dim ImColorGreen As New MCvScalar(160, 255, 160)
        Dim ImColorOrange As New MCvScalar(160, 208, 255)
        Dim ImColorRed As New MCvScalar(160, 160, 255)
        Dim ImFontScale As Double = 1
        Dim ImH As Integer = 157
        Dim ImMaxY As Integer = 1
        Dim ImOriginX As Integer = 0
        Dim ImOriginY As Integer = ImH - 32
        Dim ImRealNC As Double = (Test.MarkMax - Test.MarkMin) / CSpan
        Dim ImStepX As Integer = CInt(Int((ImW - 1) / ImRealNC))
        Dim ImLabelSize As System.Drawing.Size
        Dim ImValueSize As System.Drawing.Size
        Dim LabelSkip As Boolean

        For i = 0 To CN - 1
            If MDColumns(i) > ImMaxY Then ImMaxY = MDColumns(i)
        Next i
        Using MarksDistribution As New Image(Of Bgr, Byte)(ImW, ImH, New Bgr(255, 255, 255))
            For i = 1 To CN
                Select Case Test.MarkMin + i * CSpan
                    Case Test.MarkMax
                        ColumnLabel = "[" & FormatMark(Test.MarkMin + (i - 1) * CSpan, Test.MarksStep, "#") & "-" & FormatMark(Test.MarkMin + i * CSpan, Test.MarksStep, "#") & "]"
                    Case > Test.MarkMax
                        ColumnLabel = "[" & FormatMark(Test.MarkMin + (i - 1) * CSpan, Test.MarksStep, "#") & "-" & FormatMark(Test.MarkMax, Test.MarksStep, "#") & "]"
                    Case Else
                        ColumnLabel = "[" & FormatMark(Test.MarkMin + (i - 1) * CSpan, Test.MarksStep, "#") & "-" & FormatMark(Test.MarkMin + i * CSpan, Test.MarksStep, "#") & ")"
                End Select
                If Test.MarkMin + i * CSpan < Test.MarkPass Then
                    ImColor = ImColorRed
                ElseIf Test.MarkMin + (i - 1) * CSpan < Test.MarkPass Then
                    ImColor = ImColorOrange
                Else
                    ImColor = ImColorGreen
                End If
                ColumnValue = Format(MDColumns(i - 1))
                ImValueSize = CvInvoke.GetTextSize(ColumnValue, CvEnum.FontFace.HersheyPlain, ImFontScale, 1, 0)
                ImLabelSize = CvInvoke.GetTextSize(ColumnLabel, CvEnum.FontFace.HersheyPlain, ImFontScale, 1, 0)
                LabelSkip = (ImLabelSize.Width > ImStepX)
                CvInvoke.Line(MarksDistribution, New System.Drawing.Point(ImOriginX + (i - 1) * ImStepX, ImOriginY), New System.Drawing.Point(ImOriginX + (i - 1) * ImStepX, 0), ImColor, 1) 'vertical bars
                If i = CN Then 'last column
                    CvInvoke.Rectangle(MarksDistribution, New System.Drawing.Rectangle(ImOriginX + (CN - 1) * ImStepX, ImOriginY - CInt(MDColumns(CN - 1) * ImOriginY / ImMaxY), CInt(ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan), CInt(MDColumns(CN - 1) * ImOriginY / ImMaxY)), ImColor, -1) 'histogram
                    CvInvoke.Line(MarksDistribution, New System.Drawing.Point(ImOriginX + (CN - 1) * ImStepX, 0), New System.Drawing.Point(ImOriginX + (CN - 1) * ImStepX + CInt(ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan), 0), ImColor, 1) 'upper horizontal bar
                    If ImLabelSize.Width / 1.5 < ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan AndAlso ((Not LabelSkip) Or ((CN Mod 2) = 0)) Then CvInvoke.PutText(MarksDistribution, ColumnLabel, New System.Drawing.Point(ImOriginX + (CN - 1) * ImStepX + CInt(ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan) - ImLabelSize.Width, ImH - ImLabelSize.Height), CvEnum.FontFace.HersheyPlain, ImFontScale, ImColorBlack, 1, CvEnum.LineType.AntiAlias) 'label of horizontal axis
                    If ImValueSize.Width < ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan Then CvInvoke.PutText(MarksDistribution, ColumnValue, New System.Drawing.Point(Math.Min(ImOriginX + (CN - 1) * ImStepX + CInt(ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan) - ImValueSize.Width, ImOriginX + (CN - 1) * ImStepX + CInt(ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan - ImValueSize.Width) \ 2), Math.Min(ImOriginY - ImValueSize.Height, ImOriginY - CInt(MDColumns(CN - 1) * ImOriginY / ImMaxY) + 2 * ImValueSize.Height)), CvEnum.FontFace.HersheyPlain, ImFontScale, ImColorBlack, 1, CvEnum.LineType.AntiAlias) 'value
                Else 'other columns
                    CvInvoke.Rectangle(MarksDistribution, New System.Drawing.Rectangle(ImOriginX + (i - 1) * ImStepX, ImOriginY - CInt(MDColumns(i - 1) * ImOriginY / ImMaxY), ImStepX - 1, CInt(MDColumns(i - 1) * ImOriginY / ImMaxY)), ImColor, -1) 'histogram
                    CvInvoke.Line(MarksDistribution, New System.Drawing.Point(ImOriginX + (i - 1) * ImStepX, 0), New System.Drawing.Point(ImOriginX + (i - 1) * ImStepX + ImStepX - 1, 0), ImColor, 1) 'upper horizontal bar
                    If ImLabelSize.Width < 2 * ImStepX AndAlso ((Not LabelSkip) Or ((i Mod 2) = 0)) Then CvInvoke.PutText(MarksDistribution, ColumnLabel, New System.Drawing.Point(ImOriginX + (i - 1) * ImStepX + CInt((ImStepX - ImLabelSize.Width) / 2), ImH - ImLabelSize.Height), CvEnum.FontFace.HersheyPlain, ImFontScale, ImColorBlack, 1, CvEnum.LineType.AntiAlias) 'label of horizontal axis
                    If ImValueSize.Width < ImStepX Then CvInvoke.PutText(MarksDistribution, ColumnValue, New System.Drawing.Point(ImOriginX + (i - 1) * ImStepX + (ImStepX - ImValueSize.Width) \ 2, Math.Min(ImOriginY - ImValueSize.Height, ImOriginY - CInt(MDColumns(i - 1) * ImOriginY / ImMaxY) + 2 * ImValueSize.Height)), CvEnum.FontFace.HersheyPlain, ImFontScale, ImColorBlack, 1, CvEnum.LineType.AntiAlias) 'value
                End If
            Next i
            CvInvoke.Line(MarksDistribution, New System.Drawing.Point(ImOriginX, ImOriginY), New System.Drawing.Point(ImOriginX, 0), ImColorRed, 1) 'first vertical bar
            CvInvoke.Line(MarksDistribution, New System.Drawing.Point(ImOriginX + (CN - 1) * ImStepX + CInt(ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan), ImOriginY), New System.Drawing.Point(ImOriginX + (CN - 1) * ImStepX + CInt(ImStepX * (Test.MarkMax - Test.MarkMin - (CN - 1) * CSpan) / CSpan), 0), ImColorGreen, 1) 'last vertical bar
            Return MarksDistribution.ToBitmap
        End Using

    End Function

    Private Function CreateMessageWithNoise(ByVal Message As String) As Image(Of Gray, Byte)

        Const EF As Double = 6
        Const ImB As Integer = CInt(42 * EF)
        Const ImH As Integer = CInt(210 * EF)
        Const ImW As Integer = CInt(297 * EF)
        Const LS As Integer = CInt(EF * 9)

        Dim i As Integer
        Dim MWN As New Image(Of Gray, Byte)(ImW, ImH)
        Dim Lines As String() = Message.Split({vbNewLine}, StringSplitOptions.RemoveEmptyEntries)

        CvInvoke.Randu(MWN, New MCvScalar(0), New MCvScalar(255))
        CvInvoke.Rectangle(MWN, New System.Drawing.Rectangle(0, 0, ImW, ImB), New MCvScalar(255), -1)
        CvInvoke.Rectangle(MWN, New System.Drawing.Rectangle(0, ImH - ImB, ImW, ImB), New MCvScalar(255), -1)
        CvInvoke.Rectangle(MWN, New System.Drawing.Rectangle(0, ImB, ImB, ImH - 2 * ImB), New MCvScalar(255), -1)
        CvInvoke.Rectangle(MWN, New System.Drawing.Rectangle(ImW - ImB, ImB, ImB, ImH - 2 * ImB), New MCvScalar(255), -1)

        For i = 0 To Lines.Count - 1
            If Len(Lines(i)) > 49 Then Lines(i) = Strings.Left(Lines(i), 47) & "..."
            CvInvoke.PutText(MWN, Lines(i), New System.Drawing.Point(ImB, ImB \ 2 - CInt((Lines.Count - 1.55) * LS / 2) + i * LS), CvEnum.FontFace.HersheyPlain, EF / 2, New MCvScalar(0), Math.Max(1, CInt(EF / 4)), CvEnum.LineType.AntiAlias)
        Next i

        Return MWN

    End Function

    Private Sub CutFrame(ByRef Grey As Image(Of Gray, Byte))

        Dim BorderToCut As Integer = 0
        Dim ImgRatio As Double = Math.Max(Grey.Width, Grey.Height) / Math.Min(Grey.Width, Grey.Height)
        Dim T_Ratio As Double = T_TotalHeight / T_TotalWidth

        If ImgRatio > T_Ratio Then BorderToCut = CInt((ImgRatio - T_Ratio) * Math.Min(Grey.Width, Grey.Height) / 2)
        If BorderToCut > 0 Then
            If Grey.Height > Grey.Width Then Grey.ROI = New System.Drawing.Rectangle(0, BorderToCut, Grey.Width, Grey.Height - 2 * BorderToCut) Else Grey.ROI = New System.Drawing.Rectangle(BorderToCut, 0, Grey.Width - 2 * BorderToCut, Grey.Height)
            Grey = Grey.Copy
        End If

    End Sub

    Private Function DifferenceOfGaussians(ByRef img As Image(Of Gray, Byte)) As Image(Of Gray, Byte)

        Dim DoG As Image(Of Gray, Byte)
        Dim GaussianBlur1 As New Image(Of Gray, Byte)(img.Width, img.Height)
        Dim GaussianBlur3 As New Image(Of Gray, Byte)(img.Width, img.Height)
        Dim MeanGrey As New Image(Of Gray, Byte)(img.Width, img.Height, New Gray(128))

        CvInvoke.GaussianBlur(img, GaussianBlur1, New System.Drawing.Size(0, 0), 1)
        CvInvoke.GaussianBlur(img, GaussianBlur3, New System.Drawing.Size(0, 0), 3)
        DoG = MeanGrey + GaussianBlur1 / 2 - GaussianBlur3 / 2
        CvInvoke.GaussianBlur(DoG, DoG, New System.Drawing.Size(3, 3), 0)
        Return DoG.ThresholdBinary(New Gray(125), New Gray(255)).Not

    End Function

    Public Function DirectoryName(ByVal FullPath As String) As String

        Dim DN As String = ""
        Dim i As Integer = InStrRev(FullPath, "\", Len(FullPath) - 1)

        If i > 0 Then
            DN = Mid(FullPath, i + 1)
            If DN.EndsWith("\") Then DN = Strings.Left(DN, Len(DN) - 1)
        End If
        Return DN

    End Function

    Private Function EmguFromLaTeX(ByVal LaTex As Point) As System.Drawing.Point

        Return New System.Drawing.Point(CInt(10 * LaTex.X), CInt(10 * (T_TextHeight - LaTex.Y)))

    End Function

    Public Function FormatMark(ByVal M As Double, ByVal Mstep As Double, Optional ByVal DecimalsFormat As String = "0") As String

        Dim Decimals As Integer = InStr(Str(Mstep), ".")

        If DecimalsFormat <> "0" And DecimalsFormat <> "#" Then DecimalsFormat = "0"
        If Decimals > 0 Then Return Format(M, "0." & StrDup(Math.Min(Len(Str(Mstep)) - Decimals, 6), DecimalsFormat)) Else Return Format(M, "0")

    End Function

    Private Function GetAllAnswers(ByVal QnIndex As Integer, ByRef Grey As Image(Of Gray, Byte), Optional ByVal ShowDebug As Boolean = False) As String

        Dim Ans As Integer
        Dim Blank As New Point(0, 0) 'X mean, Y standard deviation
        Dim BlankMax As Double = 0
        Dim i As Integer
        Dim j As Integer
        Dim k As Integer = 0
        Dim Result As String = ""
        Dim SD As String = "" 'for ShowDebug
        Dim ThresholdAnswer As Double
        Dim ThresholdBlank As Double
        Dim ThresholdCancel As Double
        Dim TotAnswers As Integer = 0
        Dim V As Double()

        'total number of possible answers
        For j = 1 To Test.NumberOfQuestions
            TotAnswers += Len(Test.Contents(QnIndex).AnswersOrder(j - 1))
        Next j
        If TotAnswers < 1 Then Return "_"
        Dim VAnswer(TotAnswers - 1) As Double
        Dim VCancel(TotAnswers - 1) As Double

        'initialisations
        If ShowDebug Then Debug.Write(vbNewLine & "QnID:" & Format(Test.Contents(QnIndex).ID, "000") & " - Blank: ")
        For j = 1 To Test.NumberOfQuestions
            For i = 1 To Len(Test.Contents(QnIndex).AnswersOrder(j - 1))
                V = GetAnswersValues(j, i, Grey)
                Blank.X += V(0)
                Blank.Y += V(0) * V(0)
                If V(0) > BlankMax Then BlankMax = V(0)
                If ShowDebug Then Debug.Write(Format(V(0), "000.0") & " ")
                VAnswer(k) = V(1)
                VCancel(k) = V(2)
                k += 1
            Next i
        Next j

        'thresholds
        Blank.X /= TotAnswers
        Blank.Y = Math.Sqrt(Blank.Y / TotAnswers - Blank.X * Blank.X)
        ThresholdBlank = 1 + Math.Min(Blank.X + 3 * Blank.Y, BlankMax)
        If ShowDebug Then
            Debug.WriteLine(vbNewLine & "Blank: " & Format(Blank.X, "0.0") & " ± " & Format(Blank.Y, "0.0") & " - Max: " & Format(BlankMax, "0.0") & " - Threshold: " & Format(ThresholdBlank, "0.0"))
            SD = "Answer: "
        End If
        ThresholdAnswer = GetThreshold(VAnswer, ThresholdBlank, SD)
        If ShowDebug Then SD = "Cancel: " Else SD = ""
        ThresholdCancel = Math.Max(GetThreshold(VCancel, ThresholdBlank, SD), 24)

        'answers assignment
        For j = 1 To Test.NumberOfQuestions
            Ans = 0
            For i = 1 To Len(Test.Contents(QnIndex).AnswersOrder(j - 1))
                V = GetAnswersValues(j, i, Grey, ShowDebug)
                If V(1) > ThresholdAnswer And V(2) < ThresholdCancel Then
                    If Ans = 0 Then Ans = i Else Ans = -1
                End If
            Next i
            Select Case Ans
                Case -1
                    Result &= "0"
                Case 0
                    Result &= "_"
                Case Else
                    Result &= Mid(AnswerChars, Ans, 1)
            End Select
        Next j

        Return Result

    End Function

    Private Function GetAnswersValues(ByVal QuestionNumber As Integer, ByVal Answer As Integer, ByRef R As Image(Of Gray, Byte), Optional ByVal ShowDebug As Boolean = False) As Double()
        'returns three values: 0 Blank, 1 Answer, 2 Cancel

        Const DeltaCentres As Integer = CInt(5 * T_SquareSide + 10 * T_DeltaCancel + 10 * T_CancelRadius)
        Const InsideCancel As Integer = CInt(30 * T_CancelRadius) \ 5

        Dim Area As Integer = 0
        Dim C As System.Drawing.Point
        Dim Values As Double() = {0, 0, 0}
        Dim x As Integer
        Dim y As Integer

        C = RefineSquareCorners(EmguFromLaTeX(SquareCentre(QuestionNumber, Answer, InitialiseLaTeXData)), R)

        'mean value of white
        For y = DeltaCentres - 5 * InsideCancel \ 3 To DeltaCentres + 2 * InsideCancel
            For x = CInt(T_SquareSide * 5 - 2) To CInt(T_SquareSide * 5 + 3)
                Area += 2
                Values(0) += CInt(R.Data(C.Y + y + 2, C.X - x, 0)) + CInt(R.Data(C.Y + y + 2, C.X + x, 0))
                If ShowDebug AndAlso ((x + y) Mod 2) = 0 Then
                    R.Data(C.Y + y + 2, C.X - x, 0) = 128
                    R.Data(C.Y + y + 2, C.X + x, 0) = 128
                End If
            Next x
        Next y
        If Area > 0 Then Values(0) /= Area

        'blackened answer
        Area = 0
        For y = -CInt(T_SquareSide * 4 - 1) To CInt(T_SquareSide * 4 - 1)
            For x = -CInt(T_SquareSide * 4 - 1) To CInt(T_SquareSide * 4 - 1)
                Area += 1
                Values(1) += R.Data(C.Y + y + 2, C.X + x, 0)
                If ShowDebug AndAlso ((x + y) Mod 2) = 0 Then R.Data(C.Y + y + 2, C.X + x, 0) = 128
            Next x
        Next y
        If Area > 0 Then Values(1) /= Area

        'cancelled answer
        Area = 0
        C += New System.Drawing.Size(0, DeltaCentres)
        For y = -InsideCancel To InsideCancel
            For x = -InsideCancel To InsideCancel
                If x * x + y * y < InsideCancel * InsideCancel Then
                    Area += 1
                    Values(2) += R.Data(C.Y + y + 2, C.X + x, 0)
                    If ShowDebug AndAlso ((x + y) Mod 2) = 0 Then R.Data(C.Y + y + 2, C.X + x, 0) = 128
                End If
            Next x
        Next y
        If Area > 0 Then Values(2) /= Area

        If ShowDebug Then Debug.WriteLine(Format(QuestionNumber, "00") & "." & Answer.ToString & " - Blank:" & Format(Values(0), "00.0") & " Answer:" & Format(Values(1), "000.0") & " Cancel:" & Format(Values(2), "000.0"))
        Return Values

    End Function

    Private Function GetThreshold(ByRef V() As Double, ByVal ThresholdBlank As Double, Optional ByVal ShowDebug As String = "") As Double

        Dim DeltaV(UBound(V) - 1) As Double
        Dim i As Integer
        Dim Median As Double
        Dim Threshold As Double = ThresholdBlank * 1.5

        'initialisations
        Array.Sort(V)
        For i = 0 To UBound(V) - 1
            DeltaV(i) = V(i + 1) - V(i)
        Next i
        Array.Sort(DeltaV)
        If (UBound(DeltaV) Mod 2) = 0 Then Median = DeltaV(UBound(DeltaV) \ 2) Else Median = (DeltaV(UBound(DeltaV) \ 2) + DeltaV(UBound(DeltaV) \ 2 + 1)) / 2

        'first jump higher than ThresholdBlank
        For i = 0 To UBound(V) - 1
            DeltaV(i) = V(i + 1) - V(i)
            If V(i + 1) > ThresholdBlank AndAlso DeltaV(i) > 5 * (Median + 0.5) Then
                Threshold = (V(i) + V(i + 1)) / 2
                Exit For
            End If
        Next i

        If Not String.IsNullOrEmpty(ShowDebug) Then
            Debug.Write(ShowDebug)
            For i = 0 To UBound(V)
                Debug.Write(Format(V(i), "000.0") & " ")
            Next i
            Debug.WriteLine("- Threshold: " & Threshold)
        End If
        Return Threshold

    End Function

    Private Function GetQuestionnaireID(ByRef R As Image(Of Gray, Byte), Optional ByVal ShowDebug As Boolean = False) As Integer

        Const IDSide As Integer = CInt(((1 + 8 * T_QuestionnaireIDBits) ^ 0.5 - 1) / 2)

        Dim Area As Integer
        Dim bit As Integer = T_QuestionnaireIDBits - 1
        Dim bits(T_QuestionnaireIDBits - 1) As Double
        Dim bitsBis(T_QuestionnaireIDBits - 1) As Double
        Dim C As System.Drawing.Point
        Dim dy As Double = 0
        Dim GTID As Integer = 0
        Dim i As Integer
        Dim j As Integer
        Dim Sum As Integer
        Dim Threshold As Double
        Dim x As Integer
        Dim y As Integer

        'data collection
        For j = IDSide - 1 To 0 Step -1
            For i = IDSide - 1 To 0 Step -1
                If i + j < IDSide Then
                    C = EmguFromLaTeX(New Point(T_TextWidth - T_IDBaseShift - T_DeltaIDBits * i + 0.1, T_IDBaseShift + T_DeltaIDBits * j - 0.2))
                    Area = 0
                    Sum = 0
                    For y = -8 To 8
                        For x = -8 To 8
                            If x * x + y * y < 64 Then
                                Sum += R.Data(C.Y + y, C.X + x, 0)
                                Area += 1
                                If ShowDebug AndAlso ((x + y) Mod 2) = 0 Then R.Data(C.Y + y, C.X + x, 0) = 128
                            End If
                        Next x
                    Next y
                    If Area > 0 Then bits(bit) = Sum / Area Else bits(bit) = -1
                    bitsBis(bit) = bits(bit)
                    bit -= 1
                End If
            Next i
        Next j

        'threshold setting (highest jump) and ID evaluation
        Array.Sort(bits)
        j = -1
        For i = 0 To UBound(bits) - 1
            If bits(i + 1) - bits(i) > dy Then
                dy = bits(i + 1) - bits(i)
                j = i
            End If
        Next i
        If j >= 0 Then
            Threshold = (bits(j) + bits(j + 1)) / 2
            For i = 0 To UBound(bitsBis)
                If bitsBis(i) > Threshold Then GTID += 1 << i
            Next i
        End If

        If GTID > MaxQuestionnaireID Then Return 0
        If IndexOfQuestionnaireContent(GTID, Test) < 0 Then Return 0
        If IsNothing(Test.Contents(IndexOfQuestionnaireContent(GTID, Test))) Then Return 0
        Return GTID

    End Function

    Private Function HSCornerPixels(ByRef Grey As Image(Of Gray, Byte), ByVal C As System.Drawing.Point, ByVal Side As Integer, ByVal dx As Integer, ByVal dy As Integer, Optional ByVal ShowDebug As Boolean = False) As Double

        Const AddBorder As Integer = 8

        Dim Pixels As Integer = 0
        Dim x As Integer
        Dim y As Integer
        Dim Tot As Integer = 0

        If Side <= 0 Then Return -1
        For y = -2 - AddBorder To Side - 2 + AddBorder
            For x = -2 - AddBorder To Side - 2 + AddBorder
                If x + y < Side + AddBorder AndAlso C.X + x * dx >= 0 AndAlso C.Y + y * dy >= 0 Then
                    Pixels += 1
                    Tot += Grey.Data(C.Y + y * dy, C.X + x * dx, 0)
                    If ShowDebug AndAlso ((x + y) Mod 2) = 0 Then Grey.Data(C.Y + y * dy, C.X + x * dx, 0) = 128
                End If
            Next x
        Next y

        If ShowDebug Then
            Debug.WriteLine(C.ToString & ": " & Format(Tot / Pixels, "0.0"))
            Grey.ROI = New System.Drawing.Rectangle(C.X - 3 * Side \ 2, C.Y - 3 * Side \ 2, 3 * Side, 3 * Side)
            CvInvoke.Imshow("Corner:" & dx.ToString, Grey.Not.Resize(6, CvEnum.Inter.Area))
            Grey.ROI = Nothing
            If dx < 0 Then
                MsgBox("Press <Ok> to continue...", MsgBoxStyle.OkOnly, MainTitle)
                CvInvoke.DestroyAllWindows()
            End If
        End If

        Return Tot / Pixels

    End Function

    Private Function HSHalfPlane(ByVal P1 As System.Drawing.PointF, ByVal P2 As System.Drawing.PointF, ByVal P As System.Drawing.PointF) As Integer
        'returns 1, -1 or 0 depending on position of P with respect to P1-P2 line

        Return Math.Sign((P2.X - P1.X) * (P.Y - P1.Y) - (P2.Y - P1.Y) * (P.X - P1.X))

    End Function

    Private Function HSIntersection(ByVal R1 As System.Drawing.PointF, ByVal R2 As System.Drawing.PointF) As System.Drawing.PointF
        'intersection between the line (R1;t1) and the line (R2;t2)

        Dim Denominator As Double = Math.Sin(R1.Y - R2.Y)

        If Denominator <> 0 Then Return New System.Drawing.PointF(CSng((R2.X * Math.Sin(R1.Y) - R1.X * Math.Sin(R2.Y)) / Denominator), CSng((R1.X * Math.Cos(R2.Y) - R2.X * Math.Cos(R1.Y)) / Denominator)) Else Return New System.Drawing.PointF(Single.MinValue, Single.MinValue)

    End Function

    Private Function HSLocateGrid(ByRef DoG As Image(Of Gray, Byte), ByRef Grey As Image(Of Gray, Byte), ByVal MaskBorder As Size) As System.Drawing.PointF()

        Const IDLength As Integer = CInt(((1 + 8 * T_QuestionnaireIDBits) ^ 0.5 - 1) / 2 - 1)
        Const ShowDebug As Boolean = False

        Dim Corners As New Matrix(Of Single)(4, 2)
        Dim GetMax As Boolean = True
        Dim H As Boolean
        Dim i As Integer
        Dim j As Integer = 0
        Dim Lines As New List(Of HoughSpacePoint)
        Dim MaskedDoG As Image(Of Gray, Byte)
        Dim RawLines As New Util.VectorOfPointF 'points in HS with 0 <= Y <= Pi
        Dim RefinementHalfSize As Integer
        Dim RSides(3) As HoughSpacePoint 'sides of rectangle, horizontal ones first then vertical ones, from min to max
        Dim RV(3) As System.Drawing.PointF 'vertices of rectangle, upper ones first then lower ones, from left to right
        Dim RVold(3) As System.Drawing.PointF 'vertices of rectangle before refinement, for ShowDebug
        Dim z As Double

        'searching for lines along the four borders of the image, in up-down-left-right order
        For Each MaskRectangle As System.Drawing.Rectangle In {New System.Drawing.Rectangle(0, CInt(MaskBorder.Height * DoG.Height), DoG.Width, CInt((1 - MaskBorder.Height) * DoG.Height)), New System.Drawing.Rectangle(0, 0, DoG.Width, CInt((1 - MaskBorder.Height) * DoG.Height)), New System.Drawing.Rectangle(CInt(MaskBorder.Width * DoG.Width), 0, CInt((1 - MaskBorder.Width) * DoG.Width), DoG.Height), New System.Drawing.Rectangle(0, 0, CInt((1 - MaskBorder.Width) * DoG.Width), DoG.Height)}
            MaskedDoG = DoG.Clone
            CvInvoke.Rectangle(MaskedDoG, MaskRectangle, New MCvScalar(0), -1)
            CvInvoke.HoughLines(MaskedDoG, RawLines, 1, Math.PI / 512, CInt((1 - 2 * MaskBorder.Width) * DoG.Width / 4))
            If (Not IsNothing(RawLines)) AndAlso RawLines.Size > 0 Then
                Lines.Clear()
                For i = 0 To Math.Min(RawLines.Size - 1, 3)
                    If Not HSRawLineInLines(RawLines(i), Lines, DoG.Size) Then
                        H = (Math.Abs(PiQ(2) - RawLines(i).Y) < PiQ(1))
                        If H Xor j > 1 Then
                            If H Then z = HSIntersection(RawLines(i), New System.Drawing.PointF(DoG.Width \ 2, 0)).Y / DoG.Height Else z = HSIntersection(RawLines(i), New System.Drawing.PointF(DoG.Height \ 2, CSng(PiQ(2)))).X / DoG.Width
                            If z >= 0 And z <= 1 Then
                                If Lines.Count = 0 Then
                                    Lines.Add(New HoughSpacePoint(H, RawLines(i), z))
                                Else
                                    If Math.Abs(Lines(0).P.Y - RawLines(i).Y) > Math.PI / 64 Xor Math.Abs(Lines(0).Z - z) < 1.25 * T_Margins / Math.Min(T_TotalHeight, T_TotalWidth) Then Lines.Add(New HoughSpacePoint(H, RawLines(i), z))
                                End If
                            End If
                        End If
                    End If
                    If Lines.Count > 1 Then Exit For
                Next i
            Else
                Return Nothing
            End If
            If ShowDebug Then
                If Lines.Count > 0 Then
                    Debug.Write(vbNewLine & "Lato:" & (j + 1).ToString & vbNewLine & Lines(0).ToString)
                    If Lines.Count > 1 Then Debug.Write(Lines(1).ToString)
                End If
            End If
            Select Case Lines.Count
                Case 1
                    RSides(j) = Lines(0)
                Case 2
                    If Lines(0).Z > Lines(1).Z Then
                        If GetMax Then RSides(j) = Lines(0) Else RSides(j) = Lines(1)
                    Else
                        If GetMax Then RSides(j) = Lines(1) Else RSides(j) = Lines(0)
                    End If
                Case Else
                    Return Nothing
            End Select
            GetMax = Not GetMax
            j += 1
        Next MaskRectangle

        'vertices of rectangle
        For i = 0 To 3
            RV(i) = HSIntersection(RSides(i \ 2).P, RSides(2 + (i Mod 2)).P)
            If RV(i).X < 0 Or RV(i).Y < 0 Or RV(i).X >= DoG.Width Or RV(i).Y >= DoG.Height Then Return Nothing
        Next i

        'vertices refinement
        For i = 0 To 3
            Corners(i, 0) = RV(i).X
            Corners(i, 1) = RV(i).Y
            RVold(i) = RV(i)
        Next i
        RefinementHalfSize = Math.Max(1, Math.Min(DoG.Width, DoG.Height) \ 90)
        CvInvoke.CornerSubPix(DoG, Corners, New System.Drawing.Size(RefinementHalfSize, RefinementHalfSize), New System.Drawing.Size(-1, -1), New MCvTermCriteria(42, 0.001))
        For i = 0 To 3
            RV(i).X = Corners(i, 0)
            RV(i).Y = Corners(i, 1)
        Next i
        If HSHalfPlane(RV(0), RV(3), RV(1)) * HSHalfPlane(RV(0), RV(3), RV(2)) >= 0 Or HSHalfPlane(RV(1), RV(2), RV(0)) * HSHalfPlane(RV(1), RV(2), RV(3)) >= 0 Then Return Nothing

        'rectification and 180° rotation (if needed)
        Grey = Rectify(Grey, RV).Not
        Using RectifiedDoG As Image(Of Gray, Byte) = Rectify(DoG, RV)
            If HSCornerPixels(RectifiedDoG, EmguFromLaTeX(New Point(T_IDBaseShift, T_TextHeight - T_IDBaseShift)), 10 * CInt(T_DeltaIDBits * IDLength), 1, 1, ShowDebug) > HSCornerPixels(RectifiedDoG, EmguFromLaTeX(New Point(T_TextWidth - T_IDBaseShift, T_IDBaseShift)), 10 * CInt(T_DeltaIDBits * IDLength), -1, -1, ShowDebug) Then CvInvoke.Rotate(Grey, Grey, CvEnum.RotateFlags.Rotate180)
        End Using

        Return {RV(0), RV(1), RV(2), RV(3), RVold(0), RVold(1), RVold(2), RVold(3), New System.Drawing.PointF(RefinementHalfSize, -1)}

    End Function

    Private Function HSRawLineInLines(ByRef RawLine As System.Drawing.PointF, ByRef Lines As List(Of HoughSpacePoint), ByVal FrameSize As System.Drawing.Size) As Boolean

        Dim H As Boolean = (Math.Abs(PiQ(2) - RawLine.Y) < PiQ(1))
        Dim RLz As Double

        If H Then RLz = HSIntersection(RawLine, New System.Drawing.PointF(CSng(FrameSize.Width / 2), 0)).Y / FrameSize.Height Else RLz = HSIntersection(RawLine, New System.Drawing.PointF(CSng(FrameSize.Height / 2), CSng(PiQ(2)))).X / FrameSize.Width
        For Each Line As HoughSpacePoint In Lines
            If H = Line.Horizontal AndAlso Math.Abs(RawLine.Y - Line.P.Y) < Math.PI / 32 AndAlso Math.Abs(RLz - Line.Z) < 0.5 * T_Margins / Math.Max(T_TotalHeight, T_TotalWidth) Then Return True
        Next Line
        Return False

    End Function

    Private Sub HSShowImages(ByRef Grey As Image(Of Gray, Byte), ByRef DoG As Image(Of Gray, Byte), ByVal ID As Integer, ByVal Rotated As Boolean, ByVal GridColour As Integer, ByVal Message As String, ByVal RV() As System.Drawing.PointF, ByVal Title As String, Optional ByVal ShowDebug As Boolean = False, Optional ByVal RVold() As System.Drawing.PointF = Nothing, Optional RefinementHalfSize As Integer = 0)

        Dim ColouredDoG As Image(Of Bgr, Byte) = DoG.Convert(Of Bgr, Byte).Not
        Dim i As Integer
        Dim IDBorder As Integer
        Dim IDColour As MCvScalar
        Dim LineThickness As Integer = Math.Max(1, Math.Min(DoG.Width, DoG.Height) \ 320)
        Dim MsgScale As Double = 1
        Dim MsgSize As System.Drawing.Size
        Dim QnID As String = D("ID not recognised.")
        Dim Rendered As Image(Of Bgr, Byte)
        Dim SeparatorThickness As Integer

        If ShowDebug AndAlso (Not IsNothing(RVold)) Then
            For i = 0 To 3
                CvInvoke.Rectangle(ColouredDoG, New System.Drawing.Rectangle(CInt(RVold(i).X - RefinementHalfSize), CInt(RVold(i).Y - RefinementHalfSize), 2 * RefinementHalfSize + 1, 2 * RefinementHalfSize + 1), New MCvScalar(255, 0, 0), Math.Max(1, LineThickness \ 2))
            Next i
        End If
        If Not IsNothing(RV) Then
            For i = 0 To 3
                CvInvoke.Line(ColouredDoG, System.Drawing.Point.Round(RV(i)), System.Drawing.Point.Round(RV((i + 1) Mod 4)), New MCvScalar(0, GridColour, 255 - GridColour), LineThickness)
            Next i
        End If
        If Rotated Then 'counterstraightening
            Using CDoGUpright As New Image(Of Bgr, Byte)(ColouredDoG.Height, ColouredDoG.Width)
                CvInvoke.Rotate(ColouredDoG, CDoGUpright, CvEnum.RotateFlags.Rotate90CounterClockwise)
                ColouredDoG = CDoGUpright.Clone
            End Using
        End If
        Using RectifiedGrey As Image(Of Bgr, Byte) = Grey.Not.Convert(Of Bgr, Byte).Resize(ColouredDoG.Height / Grey.Height, CvEnum.Inter.Area)
            SeparatorThickness = (ColouredDoG.Width + RectifiedGrey.Width) \ 120
            Rendered = New Image(Of Bgr, Byte)(ColouredDoG.Width + SeparatorThickness + RectifiedGrey.Width, ColouredDoG.Height, New Bgr(0, 0, 0)) With {.ROI = New System.Drawing.Rectangle(0, 0, ColouredDoG.Width, ColouredDoG.Height)}
            ColouredDoG.CopyTo(Rendered)
            Rendered.ROI = New System.Drawing.Rectangle(ColouredDoG.Width + SeparatorThickness, 0, RectifiedGrey.Width, ColouredDoG.Height)
            RectifiedGrey.CopyTo(Rendered)
            If String.IsNullOrEmpty(Message) Then
                If ID > 0 Then QnID = D("ID:") & " " & Format(ID, "000")
            Else
                MsgSize = CvInvoke.GetTextSize(Message, CvEnum.FontFace.HersheyPlain, MsgScale, CInt(3 * MsgScale / 2), 0)
                While MsgSize.Width < 0.8 * Rendered.Width
                    MsgScale *= 1.1
                    MsgSize = CvInvoke.GetTextSize(Message, CvEnum.FontFace.HersheyPlain, MsgScale, CInt(3 * MsgScale / 2), 0)
                End While
                CvInvoke.PutText(Rendered, Message, New System.Drawing.Point(Math.Max(0, (RectifiedGrey.Width - MsgSize.Width) \ 2), Math.Max(0, (RectifiedGrey.Height - MsgSize.Height) \ 3)), CvEnum.FontFace.HersheyPlain, MsgScale, New MCvScalar(0, 0, 128), CInt(3 * MsgScale / 2))
            End If
            MsgScale = 1
            MsgSize = CvInvoke.GetTextSize(QnID, CvEnum.FontFace.HersheyPlain, MsgScale, CInt(3 * MsgScale / 2), 0)
            While MsgSize.Width < 0.6 * Rendered.Width
                MsgScale *= 1.1
                MsgSize = CvInvoke.GetTextSize(QnID, CvEnum.FontFace.HersheyPlain, MsgScale, CInt(3 * MsgScale / 2), 0)
            End While
            IDBorder = MsgSize.Width \ 16
            Rendered.ROI = New System.Drawing.Rectangle(ColouredDoG.Width + SeparatorThickness \ 2 - MsgSize.Width \ 2 - IDBorder, (Rendered.Height - MsgSize.Height) \ 2, MsgSize.Width + 2 * IDBorder, MsgSize.Height + IDBorder)
            If QnID = D("ID not recognised.") Then
                Rendered.SetValue(New Bgr(0, 0, 255))
                IDColour = New MCvScalar(0, 255, 255)
            Else
                Rendered.SetValue(New Bgr(255, 0, 0))
                IDColour = New MCvScalar(0, GridColour, 255 - GridColour)
            End If
            CvInvoke.PutText(Rendered, QnID, New System.Drawing.Point(IDBorder, MsgSize.Height + IDBorder \ 2), CvEnum.FontFace.HersheyPlain, MsgScale, IDColour, CInt(3 * MsgScale / 2))
            Rendered.ROI = Nothing
            CvInvoke.Imshow(D("Import answers from ""$$$"" - ").Replace("$$$", Options.ImagesSource).Replace("LocalImages", D("Local images")).Replace("AndroidData", D("Android device")) & Title, Rendered.Resize(MagnifyingFactor() * Math.Min(0.75 * SystemParameters.PrimaryScreenHeight / Rendered.Height, 0.75 * SystemParameters.PrimaryScreenWidth / Rendered.Width), CvEnum.Inter.Area))
        End Using

    End Sub

    Public Function IndexOfQuestionnaireContent(ByVal QID As Integer, ByRef T As TestData) As Integer

        Dim i As Integer

        For i = 0 To T.Contents.Count - 1
            If T.Contents(i).ID = QID Then Return i
        Next i
        Return -1

    End Function

    Public Function InitialiseLaTeXData() As LaTeXdata

        Dim PageData As LaTeXdata

        PageData.Columns = (Test.NumberOfQuestions - 1) \ T_MaxQuestionsPerColumn + 1
        PageData.ColumnWidth = T_NumberWidth + T_SquareSide + (Test.MaxNumberOfAnswers - 1) * T_DeltaAnswers
        PageData.DeltaColumns = PageData.ColumnWidth + 11.889
        If PageData.Columns = 1 Then
            PageData.QuestionsPerColumn = Test.NumberOfQuestions
            PageData.TotalLines = Test.NumberOfQuestions
        Else
            PageData.QuestionsPerColumn = 5 * (1 + ((Test.NumberOfQuestions - 1) \ PageData.Columns) \ 5)
            PageData.TotalLines = PageData.QuestionsPerColumn
        End If

        Return PageData

    End Function

    Private Sub InitialisePowerHistogram()

        Dim b As Double
        Dim i As Integer

        If Options.CameraFilterPowerHistogram > 0 And Options.CameraFilterPowerHistogram <> 1 Then
            For i = 0 To 255
                b = i / 255
                If b < 0.5 Then T_PowerHistogram(i, 0) = CByte(127.5 * (2 * b) ^ Options.CameraFilterPowerHistogram) Else T_PowerHistogram(i, 0) = CByte(255 - 127.5 * (2 - 2 * b) ^ Options.CameraFilterPowerHistogram)
            Next i
        End If
        T_PowerHistogramChanged = False

    End Sub

    Public Function LoadAnswersFromCamera(ByVal TestID As String, ByVal Title As String) As String()
        '0 result, 1 ID (if successful)

        Const IDMinStability As Integer = 5
        Const ShowDebug As Boolean = False

        Dim ActivationResult As String = "OK"
        Dim GL As System.Drawing.PointF()
        Dim Grey As New Image(Of Gray, Byte)(320, 200)
        Dim i As Integer
        Dim ID As Integer
        Dim IDorName As String = ""
        Dim IDStability As Integer = 0
        Dim Mask As New Image(Of Gray, Byte)(0, 0)
        Dim MaskBorder As New Size(0.2, 0.2 * T_TotalWidth / T_TotalHeight)
        Dim MaskRectangle As System.Drawing.Rectangle
        Dim OldCamera As String = ""
        Dim OldID As Integer = 0
        Dim Result As String
        Dim Rotated As Boolean = False

        For i = 0 To 255
            T_Grey4bits(i, 0) = CByte(17 * (i \ 16))
        Next i
        InitialisePowerHistogram()

        While (Not IsNothing(Grey)) And Options.ImagesSource <> "LocalImages" And Options.ImagesSource <> "AndroidData" And IDStability < IDMinStability And T_CameraStatus = CameraStatus.Running
            If Options.ImagesSource <> OldCamera Then 'change camera
                OldCamera = Options.ImagesSource
                CvInvoke.DestroyAllWindows()
                ActivationResult = ActivateCamera()
            End If
            If T_PowerHistogramChanged Then InitialisePowerHistogram()
            If ActivationResult = "OK" Then Grey = LoadFrame() Else Grey = CreateMessageWithNoise(ActivationResult)
            If Not IsNothing(Grey) Then
                ResizeTo300PPIMax(Grey)
                If Grey.Width > Grey.Height Then 'straightening
                    Using NewGreyUpright As New Image(Of Gray, Byte)(Grey.Height, Grey.Width)
                        CvInvoke.Rotate(Grey, NewGreyUpright, CvEnum.RotateFlags.Rotate90Clockwise)
                        Grey = NewGreyUpright.Clone
                    End Using
                    Rotated = True
                End If
                If Mask.Size = New System.Drawing.Size(0, 0) Then 'Mask initialisation
                    Mask = New Image(Of Gray, Byte)(Grey.Width, Grey.Height, New Gray(0))
                    MaskRectangle = New System.Drawing.Rectangle(CInt(MaskBorder.Width * Mask.Width), CInt(MaskBorder.Height * Mask.Height), CInt((1 - 2 * MaskBorder.Width) * Mask.Width), CInt((1 - 2 * MaskBorder.Height) * Mask.Height))
                    CvInvoke.Rectangle(Mask, MaskRectangle, New MCvScalar(31), -1)
                End If
                'search of grid and ID
                Using DoG As Image(Of Gray, Byte) = DifferenceOfGaussians(Grey)
                    Using OriginalDoG As Image(Of Gray, Byte) = DoG.Clone
                        CvInvoke.Rectangle(DoG, MaskRectangle, New MCvScalar(0), -1)
                        GL = HSLocateGrid(DoG, Grey, MaskBorder)
                        If IsNothing(GL) Then
                            ID = 0
                            IDStability = 0
                            Grey = New Image(Of Gray, Byte)(CInt(T_TextWidth * 10), CInt(T_TextHeight * 10), New Gray(63))
                            HSShowImages(New Image(Of Gray, Byte)(CInt(T_TextWidth * 10), CInt(T_TextHeight * 10), New Gray(63)), OriginalDoG + Mask, ID, Rotated, CInt(255 * (IDStability - 1) / (IDMinStability - 1)), D("Grid not found."), Nothing, Title, ShowDebug)
                        Else
                            ID = GetQuestionnaireID(Grey, ShowDebug)
                            If ID > 0 Then
                                If ID = OldID Then
                                    IDStability = Math.Min(IDStability + 1, IDMinStability)
                                Else
                                    IDStability = 1
                                    OldID = ID
                                End If
                            Else
                                IDStability = 0
                                OldID = 0
                            End If
                            HSShowImages(Grey, OriginalDoG + Mask, ID, Rotated, CInt(255 * (IDStability - 1) / (IDMinStability - 1)), "", {GL(0), GL(1), GL(3), GL(2)}, Title, ShowDebug, {GL(4), GL(5), GL(7), GL(6)}, CInt(GL(8).X))
                        End If
                    End Using
                End Using
                Try
                    Forms.Application.DoEvents()
                Catch ex As Exception
                End Try
                Threading.Thread.Sleep(T_Sleep \ 7)
                If IDStability = IDMinStability Then
                    If IsNothing(Test.Contents(IndexOfQuestionnaireContent(ID, Test)).Student) Then IDorName = D("ID:") & " " & Format(ID, "000") Else IDorName = D("questionnaire of ""$$$""").Replace("$$$", Test.Contents(IndexOfQuestionnaireContent(ID, Test)).Student.FamilyName & ", " & Test.Contents(IndexOfQuestionnaireContent(ID, Test)).Student.GivenName)
                    If MsgBox(D("Found $$$." & vbNewLine & "Import answers?").Replace("$$$", IDorName), MsgBoxStyle.Question Or MsgBoxStyle.OkCancel Or MsgBoxStyle.DefaultButton2, Title) = MsgBoxResult.Cancel Then IDStability = 0
                End If
            End If
        End While
        CvInvoke.DestroyAllWindows()
        If T_CameraStatus = CameraStatus.ShuttingDown Then IDStability = 0
        T_CameraStatus = CameraStatus.SwitchedOff
        If Not IsNothing(T_CaptureZ) Then T_CaptureZ.Dispose()

        If IDStability = IDMinStability Then 'answers acquisition
            Result = GetAllAnswers(IndexOfQuestionnaireContent(ID, Test), Grey, ShowDebug)
            CvInvoke.LUT(Grey, T_Grey4bits, Grey)
            Grey.Not.Save(Options.PathCourse & "ESCVtest " & TestID & "\ID" & Format(ID, "000") & ".png")
            If Test.Contents(IndexOfQuestionnaireContent(ID, Test)).GivenAnswers <> Result Then
                Test.Contents(IndexOfQuestionnaireContent(ID, Test)).GivenAnswers = Result
            End If
            Return {D("Answers imported from camera for $$$.").Replace("$$$", IDorName), Format(ID)}
        Else
            Return {D("No answers imported from camera.")}
        End If

    End Function

    Public Function LoadAnswersFromLocalImages(ByVal TestID As String) As String()
        '0 result, 1 unrecognised files (if successful)

        Const ShowDebug As Boolean = False

        Dim BadList As New List(Of String)
        Dim GL As System.Drawing.PointF()
        Dim InputPath As String
        Dim Mask As New Image(Of Gray, Byte)(0, 0)
        Dim MaskBorder As New Size(2.5 * T_Margins / T_TotalWidth, 2.5 * T_Margins / T_TotalHeight)
        Dim Pattern As String = ""
        Dim QuestionnaireID As Integer
        Dim QuestionnairesFound As Integer = 0
        Dim QuestionnairesImported As Integer = 0
        Dim R As Image(Of Gray, Byte)
        Dim Result As String

        'directory selection
        Using dlgPath As New Forms.FolderBrowserDialog With {.Description = D("Folder containing scans of answers' sheets (*.bmp, *.jpg, *.png):"), .SelectedPath = Options.PathScans, .ShowNewFolderButton = False}
            If dlgPath.ShowDialog = Forms.DialogResult.OK Then
                InputPath = dlgPath.SelectedPath
                If Not InputPath.EndsWith("\") Then InputPath &= "\"
                Options.PathScans = InputPath
            Else
                Return {""}
            End If
        End Using
        If IO.Directory.EnumerateFiles(Options.PathScans, "*.jpeg").Count > 0 Then Pattern = "*.jpeg"
        If IO.Directory.EnumerateFiles(Options.PathScans, "*.jpg").Count > 0 Then Pattern = "*.jpg"
        If IO.Directory.EnumerateFiles(Options.PathScans, "*.bmp").Count > 0 Then Pattern = "*.bmp"
        If IO.Directory.EnumerateFiles(Options.PathScans, "*.png").Count > 0 Then Pattern = "*.png"
        If String.IsNullOrEmpty(Pattern) Then Return {D("No scan of answers' sheets found in the selected folder.")}

        'answers importation
        Try
            For Each ImagePath As String In IO.Directory.EnumerateFiles(Options.PathScans, Pattern)
                'image importation
                R = New Image(Of Gray, Byte)(ImagePath)
                ResizeTo300PPIMax(R)
                If (Not IsNothing(R)) AndAlso Math.Min(R.Width, R.Height) > 412 Then
                    If R.Width > R.Height Then 'straightening (if needed)
                        Using R2 As New Image(Of Gray, Byte)(R.Height, R.Width)
                            CvInvoke.Rotate(R, R2, CvEnum.RotateFlags.Rotate90Clockwise)
                            R = R2.Clone
                        End Using
                    End If
                    'conversion and borders cleaning
                    R = R.ThresholdBinary(New Gray(Otsu(R)), New Gray(255))
                    CvInvoke.Line(R, New System.Drawing.Point(0, 0), New System.Drawing.Point(R.Width - 1, 0), New MCvScalar(255), 7)
                    CvInvoke.Line(R, New System.Drawing.Point(R.Width - 1, 0), New System.Drawing.Point(R.Width - 1, R.Height - 1), New MCvScalar(255), 7)
                    CvInvoke.Line(R, New System.Drawing.Point(R.Width - 1, R.Height - 1), New System.Drawing.Point(0, R.Height - 1), New MCvScalar(255), 7)
                    CvInvoke.Line(R, New System.Drawing.Point(0, R.Height - 1), New System.Drawing.Point(0, 0), New MCvScalar(255), 7)
                    'rectification
                    Using DoG As Image(Of Gray, Byte) = R.Clone.Not
                        GL = HSLocateGrid(DoG, R, MaskBorder)
                        If ShowDebug Then
                            Mask = New Image(Of Gray, Byte)(DoG.Width, DoG.Height, New Gray(0))
                            CvInvoke.Rectangle(Mask, New System.Drawing.Rectangle(CInt(MaskBorder.Width * Mask.Width), CInt(MaskBorder.Height * Mask.Height), CInt((1 - 2 * MaskBorder.Width) * Mask.Width), CInt((1 - 2 * MaskBorder.Height) * Mask.Height)), New MCvScalar(31), -1)
                        End If
                        'ID computation
                        If IsNothing(GL) Then
                            QuestionnaireID = 0
                            If ShowDebug Then HSShowImages(R, DoG + Mask, QuestionnaireID, False, 0, D("Grid not found."), Nothing, MainTitle, ShowDebug)
                        Else
                            QuestionnaireID = GetQuestionnaireID(R, ShowDebug)
                            If ShowDebug Then HSShowImages(R, DoG + Mask, QuestionnaireID, False, 255, "", {GL(0), GL(1), GL(3), GL(2)}, MainTitle, ShowDebug, {GL(4), GL(5), GL(7), GL(6)}, CInt(GL(8).X))
                        End If
                        If ShowDebug Then
                            MsgBox("Press <Ok> to continue...", MsgBoxStyle.OkOnly, MainTitle)
                            CvInvoke.DestroyAllWindows()
                        End If
                    End Using
                    'import and save
                    If QuestionnaireID > 0 Then
                        Result = GetAllAnswers(IndexOfQuestionnaireContent(QuestionnaireID, Test), R)
                        R.Not.Save(Options.PathCourse & "ESCVtest " & TestID & "\ID" & Format(QuestionnaireID, "000") & ".png")
                        QuestionnairesFound += 1
                        If Options.TrashRawScans Then FileIO.FileSystem.DeleteFile(ImagePath, FileIO.UIOption.OnlyErrorDialogs, FileIO.RecycleOption.SendToRecycleBin, FileIO.UICancelOption.DoNothing)
                        If Test.Contents(IndexOfQuestionnaireContent(QuestionnaireID, Test)).GivenAnswers <> Result Then
                            Test.Contents(IndexOfQuestionnaireContent(QuestionnaireID, Test)).GivenAnswers = Result
                            QuestionnairesImported += 1
                        End If
                    Else
                        BadList.Add(ChrW(34) & ImagePath & ChrW(34))
                    End If
                End If
            Next ImagePath
        Catch ex As Exception
            Beep()
            Debug.WriteLine("Beep [modEmguCV.LoadAnswersFromLocalImages]: " & ex.Message)
        End Try

        If BadList.Count = 0 Then
            Result = ""
        Else
            Result = D("It was not possible to read the ID of:")
            For Each BadFile As String In BadList
                Result &= " " & BadFile & ","
            Next BadFile
            If Result.EndsWith(",") Then Result = Strings.Left(Result, Len(Result) - 1) & "."
        End If
        Return {D("Answers' sheets imported: #I#/#F#.").Replace("#I#", Format(QuestionnairesImported)).Replace("#F#", Format(QuestionnairesFound)), Result}

    End Function

    Private Function LoadFrame() As Image(Of Gray, Byte)

        Dim NewFrame As Image(Of Gray, Byte) = Nothing

        'loading
        Try
            If Not IsNothing(T_CaptureZ) Then
                Using QF As Mat = T_CaptureZ.QueryFrame
                    If Not IsNothing(QF) Then
                        NewFrame = QF.ToImage(Of Gray, Byte)
                    Else
                        T_CaptureZ.Dispose()
                        Return CreateMessageWithNoise(D("It is not possible to get frames from:" & vbNewLine & "  ""$$$"".").Replace("$$$", Options.ImagesSource))
                    End If
                End Using
            Else
                Return CreateMessageWithNoise(D("It is not possible to get frames from:" & vbNewLine & "  ""$$$"".").Replace("$$$", Options.ImagesSource))
            End If
        Catch ex As Exception
            If Not IsNothing(My.Application) Then Return CreateMessageWithNoise(D("It is not possible to get frames from:" & vbNewLine & "  ""$$$"".").Replace("$$$", Options.ImagesSource) & vbNewLine & ex.Message)
        End Try

        'filtering
        If Not IsNothing(NewFrame) Then
            CutFrame(NewFrame)
            If Options.CameraFilterAutogamma Then Autogamma(NewFrame)
            If Options.CameraFilterStretchHistogram Then StretchHistogram(NewFrame)
            If Options.CameraFilterPowerHistogram > 0 And Options.CameraFilterPowerHistogram <> 1 Then CvInvoke.LUT(NewFrame, T_PowerHistogram, NewFrame)
        End If

        Return NewFrame

    End Function

    Private Function MagnifyingFactor() As Double

        Dim DPI As Double = Val(My.Computer.Registry.GetValue("HKEY_CURRENT_USER\SOFTWARE\Microsoft\Windows\CurrentVersion\ThemeManager", "LastLoadedDPI", "96"))
        Dim PPI As Double = Val(My.Computer.Registry.GetValue("HKEY_CURRENT_USER\SOFTWARE\Microsoft\Windows\CurrentVersion\ThemeManager", "LastLoadedPPI", "96"))

        If DPI = 0 Then DPI = 96
        If PPI = 0 Then PPI = 96
        Return DPI / PPI

    End Function

    Private Function Otsu(ByRef R As Image(Of Gray, Byte)) As Integer

        Dim Histogram(255) As Integer
        Dim i As Integer
        Dim K As Integer = 127 'default result
        Dim Max As Single = 0
        Dim Mi(255) As Single
        Dim N As Integer = R.Width * R.Height
        Dim P(255) As Single
        Dim Sigma2 As Single
        Dim Omega(255) As Single
        Dim x As Integer
        Dim y As Integer

        'initialisations
        For x = 0 To R.Width - 1
            For y = 0 To R.Height - 1
                Histogram(R.Data(y, x, 0)) += 1
            Next y
        Next x
        Omega(0) = 0
        Mi(0) = 0
        For i = 0 To 255
            P(i) = CSng(Histogram(i) / N)
            If i > 0 Then
                Omega(i) = Omega(i - 1) + P(i)
                Mi(i) = Mi(i - 1) + i * P(i)
            End If
        Next i

        'search of max of Sigma2
        For i = 0 To 255
            Sigma2 = CSng(((Mi(255) * Omega(i) - Mi(i)) ^ 2) / (Omega(i) * (1 - Omega(i))))
            If Sigma2 > Max Then
                Max = Sigma2
                K = i
            End If
        Next i

        Return K

    End Function

    Private Function Rectify(ByRef R As Image(Of Gray, Byte), ByRef G() As System.Drawing.PointF) As Image(Of Gray, Byte)

        Dim Matrix As Mat 'projection matrix
        Dim Q(3) As System.Drawing.PointF 'destination quadrilateral for projection
        Dim RQ As New Image(Of Gray, Byte)(CInt(T_TextWidth * 10), CInt(T_TextHeight * 10))

        Q(0) = New System.Drawing.PointF(0, 0)
        Q(1) = New System.Drawing.PointF(RQ.Width - 1, 0)
        Q(2) = New System.Drawing.PointF(0, RQ.Height - 1)
        Q(3) = New System.Drawing.PointF(RQ.Width - 1, RQ.Height - 1)
        Matrix = CvInvoke.GetPerspectiveTransform(G, Q)

        CvInvoke.WarpPerspective(R, RQ, Matrix, RQ.Size, CvEnum.Inter.Area)
        Return RQ

    End Function

    Private Function RefineSquareCorners(ByVal C As System.Drawing.Point, ByRef Grey As Image(Of Gray, Byte)) As System.Drawing.Point

        Dim Corners As New Matrix(Of Single)(4, 2)
        Dim i As Integer = 0
        Dim RefinementHalfSize As Integer = CInt(2 * T_SquareSide)
        Dim x As Integer
        Dim y As Integer

        For y = CInt(C.Y - T_SquareSide * 5) To CInt(C.Y + T_SquareSide * 5) Step CInt(T_SquareSide * 10)
            For x = CInt(C.X - T_SquareSide * 5) To CInt(C.X + T_SquareSide * 5) Step CInt(T_SquareSide * 10)
                Corners(i, 0) = x
                Corners(i, 1) = y
                i += 1
            Next x
        Next y
        CvInvoke.CornerSubPix(Grey, Corners, New System.Drawing.Size(RefinementHalfSize, RefinementHalfSize), New System.Drawing.Size(-1, -1), New MCvTermCriteria(42, 0.001))

        Return New System.Drawing.Point(CInt((Corners(0, 0) + Corners(1, 0) + Corners(2, 0) + Corners(3, 0)) / 4), CInt((Corners(0, 1) + Corners(1, 1) + Corners(2, 1) + Corners(3, 1)) / 4)) 'least squares

    End Function

    Public Sub ResizeTo300PPIMax(ByRef Img As Image(Of Gray, Byte))

        Dim LongerSide As Integer = Math.Max(Img.Width, Img.Height)
        Dim MaxSide As Double = T_TotalHeight * 300 / 25.4

        If LongerSide > MaxSide Then
            Img = Img.Resize(MaxSide / LongerSide, CvEnum.Inter.Area)
        End If

    End Sub

    Public Function SquareCentre(ByVal QuestionNumber As Integer, ByVal Answer As Integer, ByRef PageData As LaTeXdata) As Point

        Dim c As Integer = 1 + (QuestionNumber - 1) \ PageData.QuestionsPerColumn

        QuestionNumber = (QuestionNumber - 1) Mod PageData.QuestionsPerColumn
        Return New Point((T_TextWidth - (PageData.Columns - 1) * PageData.DeltaColumns - PageData.ColumnWidth) / 2 + (c - 1) * PageData.DeltaColumns + T_NumberWidth + T_SquareSide / 2 + (Answer - 1) * T_DeltaAnswers + T_HOffset, T_TextHeight - (T_TextHeight - PageData.TotalLines * T_DeltaQuestions) / 2 - T_DeltaQuestions / 2 - QuestionNumber * T_DeltaQuestions + T_VOffset)

    End Function

    Private Sub StretchHistogram(ByRef Grey As Image(Of Gray, Byte))

        Dim GreyMax As Double
        Dim GreyMin As Double
        Dim i As Integer
        Dim Values As New Matrix(Of Byte)(256, 1)

        CvInvoke.MinMaxLoc(Grey, GreyMin, GreyMax, New System.Drawing.Point, New System.Drawing.Point)
        For i = 0 To 255
            Values(i, 0) = CByte(Math.Max(0, Math.Min((i - GreyMin) * 255 / (GreyMax - GreyMin), 255)))
        Next i
        CvInvoke.LUT(Grey, Values, Grey)

    End Sub

End Module

Public Class HoughSpacePoint

    Public Sub New(ByVal horizontal As Boolean, ByVal p As System.Drawing.PointF, ByVal z As Double)

        Me.Horizontal = horizontal
        Me.P = p
        Me.Z = z

    End Sub

    Public Overrides Function ToString() As String

        Return "Horizontal: " & Horizontal.ToString & vbNewLine & "P: " & P.ToString & vbNewLine & "Z: " & Format(Z, "0.000") & vbNewLine

    End Function

    Public Property Horizontal() As Boolean
    Public Property P() As System.Drawing.PointF
    Public Property Z() As Double

End Class

<Serializable()> Public Class ProgramOptions

    Public Sub New()

        AveragesFormat = "0.00"
        BeginningOfYear = New System.Drawing.Point(9, 15)
        CameraFilterAutogamma = False
        CameraFilterPowerHistogram = 0
        CameraFilterStretchHistogram = False
        ClearLogOnStartup = True
        DefaultMarks = {1, 3, 6, 10}
        DefaultMarksStep = 0.1
        FTPESCVPath = "ftp://"
        FTPPassword = ""
        FTPUsername = ""
        FTPVerified = False
        HTMLindex = "index.html"
        ImagesSource = "LocalImages"
        If Globalization.CultureInfo.CurrentCulture.TwoLetterISOLanguageName = "it" Then Language = "IT" Else Language = "EN"
        LaTeXArguments = ""
        LaTeXCompilationCycles = 2
        LaTeXCompiler = "pdflatex.exe"
        LaTeXExtension = ".tex"
        LaTeXSaveSource = False
        LaTeXShowCompilation = False
        LevelsFilter = New List(Of Integer)
        ManagerColumnsWidthRatio = 1.6
        PathAssets = (Environment.CurrentDirectory & "\Assets\").Replace("\\", "\")
        PathCourse = "DefaultCourse"
        PathHTMLEditor = "notepad.exe"
        PathScans = My.Computer.FileSystem.SpecialDirectories.Desktop
        PublishAssessedOnly = True
        ReportUseColours = True
        ShowAveragesIndexes = True
        Subject = ""
        TermsAverages = True
        TermsAveragesIncludeLast = False
        TermsAveragesOnly = True
        TrashRawScans = True
        UserAddress = ""
        UserPassword = ""
        UserName = ""

    End Sub

    Public Overrides Function ToString() As String

        Dim i As Integer
        Dim ts As String = "AveragesFormat: " & AveragesFormat & vbNewLine & "BeginningOfYear: " & Format(New Date(1970, BeginningOfYear.X, 11), "MMM") & "-" & Format(BeginningOfYear.Y, "00") & vbNewLine & "CameraFilterAutogamma: " & CameraFilterAutogamma.ToString & vbNewLine & "CameraFilterPowerHistogram: " & Format(CameraFilterPowerHistogram, "0.##") & vbNewLine & "CameraFilterStretchHistogram: " & CameraFilterStretchHistogram.ToString & vbNewLine & "ClearLogOnStartup: " & ClearLogOnStartup.ToString & vbNewLine & "DefaultMarks:"

        For i = 0 To DefaultMarks.Count - 1
            ts &= Format(DefaultMarks(i), "0.000")
            If i < DefaultMarks.Count - 1 Then ts &= " - "
        Next i
        ts &= vbNewLine & "" & Format(DefaultMarksStep, "0.000") & vbNewLine & "FTPESCVPath: " & FTPESCVPath & vbNewLine & "FTPPassword: " & FTPPassword & vbNewLine & "FTPUsername: " & FTPUsername & vbNewLine & "FTPVerified: " & FTPVerified.ToString & vbNewLine & "HTMLindex: " & HTMLindex & vbNewLine & "ImagesSource: " & ImagesSource & vbNewLine & "Language: " & Language & vbNewLine & "LaTeXArguments: " & LaTeXArguments & vbNewLine & "LaTeXCompilationCycles: " & LaTeXCompilationCycles.ToString & vbNewLine & "LaTeXCompiler: " & LaTeXCompiler & vbNewLine & "LaTeXExtension: " & LaTeXExtension & vbNewLine & "LaTeXSaveSource: " & LaTeXSaveSource.ToString & vbNewLine & "LaTeXShowCompilation: " & LaTeXShowCompilation.ToString & vbNewLine & "LevelsFilter: "
        If IsNothing(LevelsFilter) Then
            ts &= "*"
        Else
            For Each V As Integer In LevelsFilter
                ts &= Format(V) & ", "
            Next V
            If ts.EndsWith(", ") Then ts = Strings.Left(ts, Len(ts) - 2)
        End If
        ts &= vbNewLine & "ManagerColumnsWidthRatio:" & Format(ManagerColumnsWidthRatio, "0.#") & "PathAssets: " & PathAssets & vbNewLine & "PathCourse: " & PathCourse & vbNewLine & "PathHTMLEditor: " & PathHTMLEditor & vbNewLine & "PathScans: " & PathScans & vbNewLine & "PublishAssessedOnly: " & PublishAssessedOnly.ToString & vbNewLine & "ReportUseColours: " & ReportUseColours.ToString & vbNewLine & "ShowAveragesIndexes: " & ShowAveragesIndexes.ToString & vbNewLine & "Subject: " & Subject & vbNewLine & "TermsAverages: " & TermsAverages.ToString & vbNewLine & "TermsAveragesIncludeLast: " & TermsAveragesIncludeLast.ToString & vbNewLine & "TermsAveragesOnly: " & TermsAveragesOnly.ToString & vbNewLine & "TrashRawScans: " & TrashRawScans.ToString & vbNewLine & "UserAddress: " & UserAddress & vbNewLine & "UserPassword: " & UserPassword & vbNewLine & "UserName: " & UserName & vbNewLine

        Return ts

    End Function

    Public Property AveragesFormat() As String
    Public Property BeginningOfYear() As System.Drawing.Point 'X month (1-12), Y day (1-31)
    Public Property CameraFilterAutogamma() As Boolean
    Public Property CameraFilterPowerHistogram() As Double
    Public Property CameraFilterStretchHistogram() As Boolean
    Public Property ClearLogOnStartup() As Boolean
    Public Property DefaultMarks() As Double()
    Public Property DefaultMarksStep() As Double
    Public Property FTPESCVPath() As String
    Public Property FTPPassword() As String
    Public Property FTPUsername() As String
    Public Property FTPVerified() As Boolean
    Public Property HTMLindex() As String
    Public Property ImagesSource() As String
    Public Property Language() As String
    Public Property LaTeXArguments() As String
    Public Property LaTeXCompilationCycles() As Integer
    Public Property LaTeXCompiler() As String
    Public Property LaTeXExtension() As String
    Public Property LaTeXSaveSource() As Boolean
    Public Property LaTeXShowCompilation() As Boolean
    Public Property LevelsFilter() As List(Of Integer)
    Public Property ManagerColumnsWidthRatio() As Double
    Public Property PathAssets() As String
    Public Property PathCourse() As String
    Public Property PathHTMLEditor() As String
    Public Property PathScans() As String
    Public Property PublishAssessedOnly() As Boolean
    Public Property ReportUseColours() As Boolean
    Public Property ShowAveragesIndexes() As Boolean
    Public Property Subject() As String
    Public Property TermsAverages() As Boolean
    Public Property TermsAveragesOnly() As Boolean
    Public Property TermsAveragesIncludeLast() As Boolean
    Public Property TrashRawScans() As Boolean
    Public Property UserAddress() As String
    Public Property UserPassword() As String
    Public Property UserName() As String

End Class

<Serializable()> Public Class QuestionData
    Implements INotifyPropertyChanged

    Private AnswersValue As List(Of String) = Nothing
    Private CategoryValue As List(Of String) = Nothing
    Private EnabledValue As Boolean = Nothing
    Private HasErrorsValue As Boolean = True
    Private IdForSerialisationValue As String = Nothing
    Private LevelValue As Integer = Nothing
    Private QuestionValue As String = Nothing

    Public Event PropertyChanged As PropertyChangedEventHandler Implements INotifyPropertyChanged.PropertyChanged
    Private Sub NotifyPropertyChanged(Optional ByVal propertyName As String = Nothing)
        RaiseEvent PropertyChanged(Me, New PropertyChangedEventArgs(propertyName))
    End Sub

    Public Sub New()

        Answers = AnswersValue
        Category = CategoryValue
        Enabled = EnabledValue
        HasErrors = HasErrorsValue
        IDForSerialisation = IdForSerialisationValue
        Level = LevelValue
        Question = QuestionValue

    End Sub

    Public Sub New(ByVal q As QuestionData)

        Answers = New List(Of String)
        For Each Ans As String In q.Answers
            Answers.Add(Ans)
        Next Ans
        Category = New List(Of String)
        For Each Cat As String In q.Category
            Category.Add(Cat)
        Next Cat
        Enabled = q.Enabled
        HasErrors = q.HasErrors
        IDForSerialisation = q.IDForSerialisation
        Level = q.Level
        Question = q.Question

    End Sub

    Public Sub New(ByVal answers As List(Of String), ByVal category As List(Of String), ByVal enabled As Boolean, ByVal hasErrors As Boolean, ByVal iDForSerialisation As String, ByVal level As Integer, ByVal question As String)

        Me.Answers = New List(Of String)
        For Each Ans As String In answers
            Me.Answers.Add(Ans)
        Next Ans
        Me.Category = New List(Of String)
        For Each Cat As String In category
            Me.Category.Add(Cat)
        Next Cat
        Me.Enabled = enabled
        Me.HasErrors = hasErrors
        Me.IDForSerialisation = iDForSerialisation
        Me.Level = level
        Me.Question = question

    End Sub

    Public Overrides Function ToString() As String

        Dim i As Integer
        Dim ts As String

        ts = "Category: "
        If Not IsNothing(Category) Then
            For i = 0 To Category.Count - 1
                ts &= Category.Item(i) & " / "
            Next i
        End If
        If ts.EndsWith(" / ") Then ts = Strings.Left(ts, Len(ts) - 3)
        ts &= vbNewLine & "Enabled: " & Enabled.ToString & vbNewLine & "HasErrors: " & HasErrors.ToString & vbNewLine & "IDForSerialisation: " & IDForSerialisation & vbNewLine & "Level: " & Level.ToString & vbNewLine & "Question: " & Question & vbNewLine
        If (Not IsNothing(Answers)) AndAlso Answers.Count > 0 Then ts &= "RightAnswer: " & Answers(0) & vbNewLine
        If (Not IsNothing(Answers)) AndAlso Answers.Count > 1 Then
            For i = 1 To Answers.Count - 1
                ts &= "WrongAnswer: " & Answers(i) & vbNewLine
            Next i
        End If

        Return ts

    End Function

    Public Shared Operator =(ByVal Q1 As QuestionData, ByVal Q2 As QuestionData) As Boolean

        Return Not (Q1 <> Q2)

    End Operator

    Public Shared Operator <>(ByVal Q1 As QuestionData, ByVal Q2 As QuestionData) As Boolean

        If Q1.Question <> Q2.Question Then Return True
        If Q1.Answers.Count <> Q2.Answers.Count Then Return True
        If Q1.Answers.Count > 0 AndAlso Q1.Answers(0) <> Q2.Answers(0) Then Return True
        For Each A As String In Q1.Answers
            If Not Q2.Answers.Contains(A) Then Return True
        Next A
        For Each B As String In Q2.Answers
            If Not Q1.Answers.Contains(B) Then Return True
        Next B
        Return False

    End Operator

    Public Property Answers() As List(Of String)
        Get
            Return AnswersValue
        End Get
        Set(ByVal value As List(Of String))
            If (IsNothing(AnswersValue) Xor IsNothing(value)) OrElse ((Not IsNothing(AnswersValue)) AndAlso (Not IsNothing(value)) AndAlso (Not AnswersValue.Equals(value))) Then
                AnswersValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Category() As List(Of String)
        Get
            Return CategoryValue
        End Get
        Set(ByVal value As List(Of String))
            If (IsNothing(CategoryValue) Xor IsNothing(value)) OrElse ((Not IsNothing(CategoryValue)) AndAlso (Not IsNothing(value)) AndAlso (Not CategoryValue.Equals(value))) Then
                CategoryValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Enabled() As Boolean
        Get
            Return EnabledValue
        End Get
        Set(ByVal value As Boolean)
            If EnabledValue <> value Then
                EnabledValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property HasErrors() As Boolean
        Get
            Return HasErrorsValue
        End Get
        Set(ByVal value As Boolean)
            If HasErrorsValue <> value Then
                HasErrorsValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property IDForSerialisation() As String
        Get
            Return IdForSerialisationValue
        End Get
        Set(ByVal value As String)
            If IdForSerialisationValue <> value Then
                IdForSerialisationValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Level() As Integer
        Get
            Return LevelValue
        End Get
        Set(ByVal value As Integer)
            If LevelValue <> value Then
                LevelValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Question() As String
        Get
            Return QuestionValue
        End Get
        Set(ByVal value As String)
            If QuestionValue <> value Then
                QuestionValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

End Class

<Serializable()> Public Class QuestionnaireContents
    Implements INotifyPropertyChanged

    Private ActualCompensationValue As Integer = 0
    Private AnswersOrderValue As New List(Of String)
    Private AssessmentValue As Integer = 0
    Private GivenAnswersValue As String = ""
    Private IDValue As Integer = Nothing
    Private ManualMarkValue As Boolean = False
    Private MarkValue As String = Nothing
    Private NumberOfAnswersEmptyValue As Integer = Nothing
    Private NumberOfAnswersNOValue As Integer = Nothing
    Private NumberOfAnswersOKValue As Integer = Nothing
    Private PenaltyValue As Integer = 0
    Private PenaltyCauseValue As String = ""
    Private PointsValue As Integer = Nothing
    Private PointsEmptyValue As Integer = Nothing
    Private PointsMaxValue As Integer = Nothing
    Private PointsPassValue As Integer = Nothing
    Private QuestionsIDValue As New List(Of String)
    Private RightAnswersValue As String = ""
    Private StudentValue As StudentData = Nothing

    Public Event PropertyChanged As PropertyChangedEventHandler Implements INotifyPropertyChanged.PropertyChanged
    Private Sub NotifyPropertyChanged(Optional ByVal propertyName As String = Nothing)
        RaiseEvent PropertyChanged(Me, New PropertyChangedEventArgs(propertyName))
    End Sub

    Public Sub New()

        ActualCompensation = ActualCompensationValue
        AnswersOrder = AnswersOrderValue
        Assessment = AssessmentValue
        GivenAnswers = GivenAnswersValue
        ID = IDValue
        ManualMark = ManualMarkValue
        Mark = MarkValue
        NumberOfAnswersEmpty = NumberOfAnswersEmptyValue
        NumberOfAnswersNO = NumberOfAnswersNOValue
        NumberOfAnswersOK = NumberOfAnswersOKValue
        Penalty = PenaltyValue
        PenaltyCause = PenaltyCauseValue
        Points = PointsValue
        PointsEmpty = PointsEmptyValue
        PointsMax = PointsMaxValue
        PointsPass = PointsPassValue
        QuestionsID = QuestionsIDValue
        RightAnswers = RightAnswersValue
        Student = StudentValue

    End Sub

    Public Overrides Function ToString() As String

        Dim f As String
        Dim i As Integer
        Dim ts As String = "ActualCompensation: " & ActualCompensation.ToString & vbNewLine & "AnswersOrder: "

        If AnswersOrder.Count = 0 Then
            ts &= vbNewLine
        Else
            f = StrDup(Len(Format(AnswersOrder.Count)), "0")
            For i = 0 To AnswersOrder.Count - 1
                If i > 0 Then ts &= Space(14)
                ts &= Format(i, f) & ". " & AnswersOrder(i) & vbNewLine
            Next i
        End If
        ts &= "Assessment: " & Assessment.ToString & vbNewLine & "GivenAnswers: " & GivenAnswers & vbNewLine & "ID: " & ID.ToString & vbNewLine & "ManualMark: " & ManualMark.ToString & vbNewLine & "Mark: " & Format(Mark, "0.00") & vbNewLine & "NumberOfAnswersEmpty: " & NumberOfAnswersEmpty.ToString & vbNewLine & "NumberOfAnswersNO: " & NumberOfAnswersNO.ToString & vbNewLine & "NumberOfAnswersOK: " & NumberOfAnswersOK.ToString & vbNewLine & "Penalty: " & Penalty.ToString & vbNewLine & "PenaltyCause: " & PenaltyCause.ToString & vbNewLine & "Points: " & Points.ToString & vbNewLine & "PointsEmpty: " & PointsEmpty.ToString & vbNewLine & "PointsPass: " & PointsPass.ToString & vbNewLine & "PointsMax: " & PointsMax.ToString & vbNewLine & "QuestionsID: [" & QuestionsID.Count.ToString & "]" & vbNewLine & "RightAnswers: " & RightAnswers & vbNewLine & "  Student:" & vbNewLine
        If Not IsNothing(Student) Then ts &= Student.ToString

        Return ts

    End Function

    Public Property ActualCompensation() As Integer
        Get
            Return ActualCompensationValue
        End Get
        Set(ByVal value As Integer)
            If ActualCompensationValue <> value Then
                ActualCompensationValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property AnswersOrder() As List(Of String)
        Get
            Return AnswersOrderValue
        End Get
        Set(ByVal value As List(Of String))
            If (IsNothing(AnswersOrderValue) Xor IsNothing(value)) OrElse ((Not IsNothing(AnswersOrderValue)) AndAlso (Not IsNothing(value)) AndAlso (Not AnswersOrderValue.Equals(value))) Then
                AnswersOrderValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Assessment() As Integer '-2 very below pass mark, -1 slightly below pass mark, 0 not assessed, 1 ok
        Get
            Return AssessmentValue
        End Get
        Set(ByVal value As Integer)
            If AssessmentValue <> value Then
                AssessmentValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property GivenAnswers() As String
        Get
            Return GivenAnswersValue
        End Get
        Set(ByVal value As String)
            If GivenAnswersValue <> value Then
                GivenAnswersValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property ID() As Integer
        Get
            Return IDValue
        End Get
        Set(ByVal value As Integer)
            If IDValue <> value Then
                IDValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property ManualMark() As Boolean
        Get
            Return ManualMarkValue
        End Get
        Set(ByVal value As Boolean)
            If ManualMarkValue <> value Then
                ManualMarkValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Mark() As String
        Get
            Return MarkValue
        End Get
        Set(ByVal value As String)
            If MarkValue <> value Then
                MarkValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property NumberOfAnswersEmpty() As Integer
        Get
            Return NumberOfAnswersEmptyValue
        End Get
        Set(ByVal value As Integer)
            If NumberOfAnswersEmptyValue <> value Then
                NumberOfAnswersEmptyValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property NumberOfAnswersNO() As Integer
        Get
            Return NumberOfAnswersNOValue
        End Get
        Set(ByVal value As Integer)
            If NumberOfAnswersNOValue <> value Then
                NumberOfAnswersNOValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property NumberOfAnswersOK() As Integer
        Get
            Return NumberOfAnswersOKValue
        End Get
        Set(ByVal value As Integer)
            If NumberOfAnswersOKValue <> value Then
                NumberOfAnswersOKValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Penalty() As Integer
        Get
            Return PenaltyValue
        End Get
        Set(ByVal value As Integer)
            If PenaltyValue <> value Then
                PenaltyValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property PenaltyCause() As String
        Get
            Return PenaltyCauseValue
        End Get
        Set(ByVal value As String)
            If PenaltyCauseValue <> value Then
                PenaltyCauseValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Points() As Integer
        Get
            Return PointsValue
        End Get
        Set(ByVal value As Integer)
            If PointsValue <> value Then
                PointsValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property PointsEmpty() As Integer
        Get
            Return PointsEmptyValue
        End Get
        Set(ByVal value As Integer)
            If PointsEmptyValue <> value Then
                PointsEmptyValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property PointsMax() As Integer
        Get
            Return PointsMaxValue
        End Get
        Set(ByVal value As Integer)
            If PointsMaxValue <> value Then
                PointsMaxValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property PointsPass() As Integer
        Get
            Return PointsPassValue
        End Get
        Set(ByVal value As Integer)
            If PointsPassValue <> value Then
                PointsPassValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property QuestionsID() As List(Of String)
        Get
            Return QuestionsIDValue
        End Get
        Set(ByVal value As List(Of String))
            If (IsNothing(QuestionsIDValue) Xor IsNothing(value)) OrElse ((Not IsNothing(QuestionsIDValue)) AndAlso (Not IsNothing(value)) AndAlso (Not QuestionsIDValue.Equals(value))) Then
                QuestionsIDValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property RightAnswers() As String
        Get
            Return RightAnswersValue
        End Get
        Set(ByVal value As String)
            If RightAnswersValue <> value Then
                RightAnswersValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

    Public Property Student() As StudentData
        Get
            Return StudentValue
        End Get
        Set(ByVal value As StudentData)
            If (IsNothing(StudentValue) Xor IsNothing(value)) OrElse ((Not IsNothing(StudentValue)) AndAlso (Not IsNothing(value)) AndAlso (Not StudentValue.Equals(value))) Then
                StudentValue = value
                NotifyPropertyChanged()
            End If
        End Set
    End Property

End Class

<Serializable()> Public Class QuestionsInCategory

    Public Sub New()

        CategoryWithLevel = ""
        Total = 0
        Used = 0

    End Sub

    Public Sub New(ByVal categoryWithLevel As String, ByVal total As Integer, ByVal used As Integer)

        Me.CategoryWithLevel = categoryWithLevel
        Me.Total = total
        Me.Used = used

    End Sub

    Public Overrides Function ToString() As String

        Return "CategoryWithLevel: " & CategoryWithLevel & vbNewLine & "Total: " & Total.ToString & vbNewLine & "Used: " & Used.ToString & vbNewLine

    End Function

    Public Property CategoryWithLevel() As String
    Public Property Total() As Integer
    Public Property Used() As Integer

End Class

<Serializable()> Public Class StudentData

    Public Sub New()

        Compensation = 0
        FamilyName = ""
        GivenName = ""
        Withdrawn = False

    End Sub

    Public Sub New(ByVal compensation As Double, familyName As String, givenName As String, withdrawn As Boolean)

        Me.Compensation = compensation
        Me.FamilyName = familyName
        Me.GivenName = givenName
        Me.Withdrawn = withdrawn

    End Sub

    Public Overrides Function ToString() As String

        Return "Compensation: " & Compensation.ToString & vbNewLine & "FamilyName: " & FamilyName & vbNewLine & "GivenName: " & GivenName & vbNewLine & "Withdrawn: " & Withdrawn.ToString & vbNewLine

    End Function

    Public Property Compensation() As Double
    Public Property FamilyName() As String
    Public Property GivenName() As String
    Public Property Withdrawn() As Boolean

End Class

<Serializable()> Public Class TestData

    Public Sub New()

        AddFrontMatter = True
        AssessLevel = False
        CategoryLengths = New List(Of Integer)
        CategoryList = New ObservableCollection(Of QuestionsInCategory)
        Contents = New ObservableCollection(Of QuestionnaireContents)
        IgnoreLevels = False
        KeepCategoriesSeparate = False
        MarkEmpty = 3
        MarkMax = 10
        MarkMin = 1
        MarkPass = 6
        MarksDistributionColumnsSpan = 1
        MarksStep = 0.1
        MaxNumberOfAnswers = -1
        NumberOfQuestions = 0
        NumberOfQuestionnairesToBeCreated = 1
        NumberOfAnswers = Integer.MinValue
        QuestionnairesAssessed = False
        QuestionnairesCreated = False
        Questions = New List(Of QuestionData)
        SameQuestionsForEveryone = False
        Students = New List(Of StudentData)
        Subject = ""
        Title = ""
        UnnamedCompensatedQuestionnaires = 0
        UnnamedCompensation = 0.3
        UseAutomaticGrid = True
        UseNames = True
        Year = ""

    End Sub

    Public Overrides Function ToString() As String

        Dim ts As String = "AddFrontMatter: " & AddFrontMatter.ToString & vbNewLine & "AssessLevel: " & AssessLevel.ToString & vbNewLine & "CategoryLengths: [" & CategoryLengths.Count.ToString & "]" & vbNewLine & "CategoryList: " & CategoryList.Count & vbNewLine & "Contents: [" & Contents.Count.ToString & "]" & vbNewLine & "IgnoreLevels: " & IgnoreLevels.ToString & vbNewLine & "KeepCategoriesSeparate: " & KeepCategoriesSeparate.ToString & vbNewLine & "MarkMin :" & FormatMark(MarkMin, MarksStep) & vbNewLine & "MarkEmpty :" & FormatMark(MarkEmpty, MarksStep) & vbNewLine & "MarkPass :" & FormatMark(MarkPass, MarksStep) & vbNewLine & "MarkMax :" & FormatMark(MarkMax, MarksStep) & vbNewLine & "MarksDistributionColumnsSpan: " & MarksDistributionColumnsSpan.ToString & vbNewLine & "MarksStep :" & MarksStep.ToString & vbNewLine & "MaxNumberOfAnswers: " & MaxNumberOfAnswers.ToString & vbNewLine & "NumberOfQuestions: " & NumberOfQuestions.ToString & vbNewLine & "NumberOfQuestionnairesToBeCreated: " & NumberOfQuestionnairesToBeCreated.ToString & vbNewLine & "NumberOfAnswers: " & NumberOfAnswers.ToString & vbNewLine & "QuestionnairesAssessed: " & QuestionnairesAssessed.ToString & vbNewLine & "QuestionnairesCreated: " & QuestionnairesCreated.ToString & vbNewLine & "Questions: [" & Questions.Count.ToString & "]" & vbNewLine & "SameQuestionsForEveryone: " & SameQuestionsForEveryone.ToString & vbNewLine & "Students: " & Students.Count
        Dim W As Integer = 0

        For Each S As StudentData In Students
            If S.Withdrawn Then W += 1
        Next S
        If W > 0 Then ts &= " - " & Format(W)
        ts &= vbNewLine & "Subject: " & Subject & vbNewLine & "Title: " & Title & vbNewLine & "UnnamedCompensatedQuestionnaires: " & UnnamedCompensatedQuestionnaires.ToString & vbNewLine & "UnnamedCompensation: " & UnnamedCompensation.ToString & vbNewLine & "UseAutomaticGrid: " & UseAutomaticGrid.ToString & vbNewLine & "UseNames: " & UseNames.ToString & vbNewLine & "Year: " & Year & vbNewLine

        Return ts

    End Function

    Public Property AddFrontMatter() As Boolean
    Public Property AssessLevel() As Boolean
    Public Property CategoryLengths() As List(Of Integer)
    Public Property CategoryList() As ObservableCollection(Of QuestionsInCategory)
    Public Property Contents() As ObservableCollection(Of QuestionnaireContents)
    Public Property IgnoreLevels() As Boolean
    Public Property KeepCategoriesSeparate() As Boolean
    Public Property MarkEmpty() As Double
    Public Property MarkMax() As Double
    Public Property MarkMin() As Double
    Public Property MarkPass() As Double
    Public Property MarksDistributionColumnsSpan() As Double
    Public Property MarksStep() As Double
    Public Property MaxNumberOfAnswers() As Integer
    Public Property NumberOfQuestions() As Integer
    Public Property NumberOfQuestionnairesToBeCreated() As Integer
    Public Property NumberOfAnswers() As Integer
    Public Property QuestionnairesAssessed() As Boolean
    Public Property QuestionnairesCreated() As Boolean
    Public Property Questions() As List(Of QuestionData)
    Public Property SameQuestionsForEveryone() As Boolean
    Public Property Students() As List(Of StudentData)
    Public Property Subject() As String
    Public Property Title() As String
    Public Property UnnamedCompensatedQuestionnaires() As Integer
    Public Property UnnamedCompensation() As Double
    Public Property UseAutomaticGrid() As Boolean
    Public Property UseNames() As Boolean
    Public Property Year() As String

End Class
